#!/bin/bash
# Helper script to install USB device rules for sigrok/PulseView

echo "Installing USB device rules for FREE-WILi logic analyzer..."

# Check for all required udev rules files
missing_files=()
for file in "60-libsigrok.rules" "61-libsigrok-plugdev.rules" "61-libsigrok-uaccess.rules"; do
    if [ ! -f "$file" ]; then
        missing_files+=("$file")
    fi
done

if [ ${#missing_files[@]} -ne 0 ]; then
    echo "Error: Missing udev rules files:"
    printf '  %s\n' "${missing_files[@]}"
    echo "Please run this script from the same directory as your AppImage"
    exit 1
fi

echo "Installing udev rules (requires sudo)..."
sudo cp 60-libsigrok.rules 61-libsigrok-plugdev.rules 61-libsigrok-uaccess.rules /etc/udev/rules.d/
if [ $? -eq 0 ]; then
    echo "All udev rules copied successfully"
else
    echo "Failed to copy rules"
    exit 1
fi

echo "Reloading udev rules..."
sudo udevadm control --reload-rules
sudo udevadm trigger

# Check for FTDI driver conflicts and provide guidance
echo ""
echo "Checking for FTDI driver conflicts..."
if dmesg 2>/dev/null | grep -q "ftdi_sio.*FTDI USB Serial Device converter detected" || lsmod | grep -q "ftdi_sio"; then
    echo "IMPORTANT: The ftdi_sio kernel driver may interfere with your logic analyzer."
    echo "This is a common issue when using FTDI-based devices for logic analysis."
    echo ""
    echo "If your device is not detected by PulseView, try this safe fix:"
    echo "   sudo modprobe -r ftdi_sio"
    echo "   (This temporarily removes the driver until next reboot)"
    echo ""
    echo "For permanent solution (optional, for dedicated development machines):"
    echo "   echo 'blacklist ftdi_sio' | sudo tee -a /etc/modprobe.d/blacklist-ftdi.conf"
    echo "   sudo update-initramfs -u && reboot"
    echo ""
else
    echo "No FTDI driver conflicts detected."
fi

echo ""
echo "USB device rules installed successfully!"
echo "Please unplug and replug your logic analyzer"
echo "Your FREE-WILi device should now be detected by PulseView"
echo ""
